#include "numerics.h"                                                           //numerické metody

#include "glob_prom.h"                                                          //globální proměnné
#include "postfix.h"                                                            //knihovna pro práci s postfixem
#include "operators.h"                                                          //přetížení operátorů pro čtyřvektory

QString to_time(qint64 t)                                                       //převod tisícin sekundy na čas
{
    int d = t / 86400000;
    int h = t / 3600000 % 24;
    int m = t / 60000 % 60;
    int s = t / 1000 % 60;
    return QString("%1:%2:%3:%4")
        .arg(d, 2, 10, QChar('0'))
        .arg(h, 2, 10, QChar('0'))
        .arg(m, 2, 10, QChar('0'))
        .arg(s, 2, 10, QChar('0'));
}

real pw(real x, int n)                                                          //celočíselná mocnina
{
    switch (n) {
    case 0: return 1;
    case 1: return x;                                                           //pro malé mocniny počítej přímo
    case 2: return x * x;
    case 3: return x * x * x;
    default:
       real result = x;
       for (int i = 1; i < n; ++i) result *= x;
       return result;
    };
};

real der(const uchar (&g)[arl], const char &γ, f_vec b)                         //numerická derivace postfixu
{
    f_vec a = b;
    a[γ] -= ε;
    b[γ] += ε;
    return (CntPfx(g, b) - CntPfx(g, a)) * jl2ε;
}

f_vec rungekuta(f_vec (*f)(f_vec u, tensor_3 &c), f_vec u, tensor_3 c, real h)  //řešení ODR rk4
{
    h /= 2.;
    f_vec k1 = (*f)(u, c), k2 = (*f)(u + k1 * h, c), k3 = (*f)(u + k2 * h, c);
    return u + (k1 + (k2 + k3) * 2. + (*f)(u + k3 * h, c)) * h / 3.;
}

void RGBtoHSL(int red, int green, int blue, real& hue, real& satur, real& light)//převod barev na HSL
{
    real r = red / 255., g = green / 255., b = blue / 255.;
    real max = fmax(fmax(r, g), b), min = fmin(fmin(r, g), b);
    light = (max + min) / 2;
    if (max == min) hue = satur = 0;
    else {
        real delta = max - min;
        satur = (light > .5) ? (delta / (2 - max - min)) : (delta / (max + min));
        if (max == r)      hue = 60 * fmod(((g - b) / delta), 6);
        else if (max == g) hue = 60 * (((b - r) / delta) + 2);
        else               hue = 60 * (((r - g) / delta) + 4);
        if (hue < 0) hue += 360;
    }
}

void HSLtoRGB(real hue, real sat, real li, int& red, int& green, int& blue)     //převod HSL na barvy
{
    if (sat == 0.0) red = green = blue = static_cast<int>(li * 255.0);
    else {
        real q = (li < .5) ? (li * (1 + sat)) : (li + sat - li * sat);
        real p = 2 * li - q;
        hue /= 360;
        auto hue2rgb = [](real p, real q, real t) -> real {
            if (t < 0) t++;
            if (t > 1) t--;
            if (t < 1. / 6) return p + (q - p) * 6 * t;
            if (t < 1. / 2) return q;
            if (t < 2. / 3) return p + (q - p) * (2. / 3 - t) * 6;
            return p;
        };
        red =   int(hue2rgb(p, q, hue + 1. / 3) * 255);
        green = int(hue2rgb(p, q, hue) * 255);
        blue =  int(hue2rgb(p, q, hue - 1. / 3) * 255);
    }
}

real l_pol(real x, int r, int g, int b)                                         //Lagrangeův polynom 11 řádu
{
    return
        (((5 * r * (x - 2) - 6 * g * (x - 1)) * (x - 3) + 5 * b * (x - 2) *
        (x - 1)) * (x - 7) * (x - 6) * (x - 5) * (x - 4) * x * (1 + x) *
        (2 + x) * (3 + x)) / 86400;
}

uint32_t sh_col(uint32_t col, real a)                                           //posun barev o redshift a
{
    int alpha = (col & 0xff000000) >> 24;
    int red = (col & 0x00ff0000) >> 16;
    int green = (col & 0x0000ff00) >> 8;
    int blue = col & 0x000000ff;
    if (simul){                                                                 //rotací spektra
        real hue, satur, light;
        RGBtoHSL(red, green, blue, hue, satur, light);
        hue -= a * 180;
        while (hue < 0.0) hue += 360;
        while (hue >= 360) hue -= 360;
        HSLtoRGB(hue, satur, light, red, green, blue);
    }
    else {                                                                      //posunem spektra
        real x = tanh(a) * 2;
        int rc = l_pol(x + 1,red,green,blue) * (x + 1);
        int gc = l_pol(x + 2,red,green,blue);
        int bc = l_pol(x + 3,red,green,blue) * (1 - x);
        red = std::clamp((rc * 5 + red) / 6, 0, 255);                           //nechám šestinu původní barvy
        green = std::clamp((gc * 5 + green) / 6, 0, 255);
        blue = std::clamp((bc * 5 + blue) / 6, 0, 255);
    }
    uint32_t reColor = alpha << 24 | red << 16 | green << 8 | blue;
    return reColor;
}

uint32_t bm_col(uint32_t col, real δ)                                           //zjasnění barev o beaming δ
{
    δ = (δ < 0) ? (1 / tanh(1)) : (tanh(δ) / tanh(1));                          //komprimace dynamiky jasů
    int alpha = (col & 0xff000000) >> 24;
    int red = (col & 0x00ff0000) >> 16;
    int green = (col & 0x0000ff00) >> 8;
    int blue = col & 0x000000ff;
    red = fmin(red * δ, 255);
    green = fmin(green * δ, 255);
    blue = fmin(blue * δ, 255);
    uint32_t reColor = alpha << 24 | red << 16 | green << 8 | blue;
    return reColor;
}


uint32_t av_col(uint32_t col1, uint32_t col2) {
    uint8_t a1 = col1 >> 24;                                                    //rozloření barev a bajty
    uint8_t r1 = (col1 >> 16) & 0xFF;
    uint8_t g1 = (col1 >> 8) & 0xFF;
    uint8_t b1 = col1 & 0xFF;
    uint8_t a2 = col2 >> 24;
    uint8_t r2 = (col2 >> 16) & 0xFF;
    uint8_t g2 = (col2 >> 8) & 0xFF;
    uint8_t b2 = col2 & 0xFF;
    int tr_a2 = transp * a2;
    int sum_a = a1 + tr_a2;
    uint8_t avgR = (a1 * r1 + tr_a2 * r2) / sum_a;                              //vážený průměr
    uint8_t avgG = (a1 * g1 + tr_a2 * g2) / sum_a;
    uint8_t avgB = (a1 * b1 + tr_a2 * b2) / sum_a;
    uint32_t averageColor = avgR << 16 | avgG << 8 | avgB | 0xFF000000;
    return averageColor;
}

void invMatrix(real (&m)[4][4]) {                                               //výpočet inverní matice
    real mat[4][8];
    for (int i = 0; i < 4; i++)                                                 //vytvoří rozšířenou matici
        for (int j = 0; j < 4; j++)
            mat[i][j] = m[i][j];
    for (int i = 0; i < 4; i++)                                                 //inicializace inverzní matice jako matice identity
        for (int j = 4; j < 8; j++)
            mat[i][j] = (i == (j-4)) ? 1 : 0;
    for (int i = 0; i < 4; i++) {                                               //provádí Gaussovu-Jordanovu eliminaci
        int maxRow = i;                                                         //hledá maximální prvek
        for (int k = i + 1; k < 4; k++)
            if (abs(mat[k][i]) > abs(mat[maxRow][i]))
                maxRow = k;
        if (i != maxRow)                                                        //výměna maximálního řádku za aktuální řádek
            for (int k = 0; k < 8; k++) {
                real temp = mat[i][k];
                mat[i][k] = mat[maxRow][k];
                mat[maxRow][k] = temp;
            }
        for (int k = 0; k < 4; k++)
            if (k != i) {
                real ratio = mat[k][i] / mat[i][i];
                for (int j = 0; j < 8; j++)
                    mat[k][j] -= ratio * mat[i][j];
            }
    }
    for (int i = 0; i < 4; i++) {                                               //normalizace diagonály
        real a = mat[i][i];
        for (int j = 0; j < 8; j++)
            mat[i][j] /= a;
    }
    for (int i = 0; i < 4; i++)
        for (int j = 0; j < 4; j++)
            m[i][j] = mat[i][j+4];
}

real dot_p(real (&m)[4], real (&n)[4], f_vec x)                                 //skalární součin v g_{μν}
{
    real dot = 0;
    for (int k = 0; k < 4; ++k)
        for (int l = 0; l < 4; ++l)
            dot += CntPfx(t.kov[k][l].n, x) * m[k] * n[l];
    return dot;
};

void norm(real (&v)[4], f_vec x)                                                //normalizace čtyřvektoru v g_{μν}
{
    real vel = sqrt(abs(dot_p(v, v, x)));
    for (int i = 0; i < 4; ++i)
        v[i] /=  vel;
}

void ortonormalize(real (&m)[4][4], f_vec x) {                                  //Gram-Smidtova ortogonalizace
    for (int i = 1; i < 4; ++i){                                                //Pro každý řádek kromě prvního projdeme všechny předchozí řádky
        for (int j = 0; j < i; ++j) {                                           //Vypočítáme skalární součin mezi i-tým a j-tým řádkem
            real s = dot_p(m[i], m[j], x);
            for (int k = 0; k < 4; ++k)                                         //Od i-tého řádku odečteme projekci na j-tý řádek
                m[i][k] -= s * m[j][k];
        }
        norm(m[i], x);                                                          //výsledný řádek normalizujeme
    }
}
